{****************************************************************************************
 * idt.pp - Sets up our IDT and contains the interrupt handlers
 *
 * written by Bastian Gloeckle (MrSaint), programmer of nucleOS
 *
 * Copyright (C) 2004 by nucleOS group
 *
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation
 * (version 2, June 1991)
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program; if not, write to:
 * Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330
 * Boston, MA 02111-1307 USA
 *
 *
 * You can contact me by electronic mail: admin@saint-soft.de
 ****************************************************************************************}
unit idt;

interface

uses mm, gfxdrvunit, convert, bits;

procedure init_idt;
procedure set_interrupt( Interrupt: byte; Address: Pointer);

//procedure setup_idt; external;
procedure setup_idt(unhandled, int21: Pointer); external;

procedure writeln(value: Pointer); external;

var
  bGFXDRVCurrentX: byte; cvar; external;
  bGFXDRVCurrentY: byte; cvar; external;
  IDTBASE:           PLongInt; cvar; external;

type
    int_gate = record
            offset_low:  Word;
            selector:    Word;
            access:      Word;
            offset_high: Word;
    end;



implementation


procedure _int21h;[public, alias:'_INT21H']; interrupt;				{ "interrupt" --> FPC documentation -> "prog.pdf" v1.9 -> Table 6.3 }
var bFunc: byte;

	procedure _int21h_write;
	var bLength, bX, bY: byte;
	begin
	  asm
	    push ds				{ get length of string }
	    push fs
	    pop ds
	    lodsb
	    pop ds
	    dec esi
	    mov [bLength], al
	  end;
	  bX := bGFXDRVCurrentX;				{ save output coordinates }
	  bY := bGFXDRVCurrentY;
	  if bGFXDRVCurrentX+bLength > 79 then			{ set bCurrentX and bCurrentY correct }
	  begin
	    bGFXDRVCurrentY += (bGFXDRVCurrentX+bLength) div 79;	{ if the text is longer than the line, we have to adjust bGFXDRVCurrentY }
	  end;
	  while bGFXDRVCurrentY>24 do				{ perhaps we need to scroll? }
	  begin
	    GFXDRVScrollOneLine;
	    dec(bGFXDRVCurrentY);
	  end;
	  bGFXDRVCurrentX := (bGFXDRVCurrentX+bLength) mod 79;	{ set bGFXDRVCurrentX }
	  GFXDRVwrite(bX,bY,$7);				{ output the string }
	end;

begin
  asm
    mov [bFunc], ah
    push fs				{ save FS and set it and DS correct }
    mov ax, ds
    mov fs, ax
    mov ax, 10h
    mov ds, ax
  end;
  case bFunc of
    01: begin	{ WriteLn }
          _int21h_write;		{ output text }
	  bGFXDRVCurrentX := 0;
	  inc(bGFXDRVCurrentY);		{ go one line down }
	  if bGFXDRVCurrentY > 24 then	{ do we need to scroll? }
	  begin
	    GFXDRVScrollOneLine;
	    bGFXDRVCurrentY := 24;
	  end;
          GFXDRVSetCursorPos(bGFXDRVCurrentX, bGFXDRVCurrentY);	{ set the cursor accordingly }
	end;
    02: begin	{ Write }
          _int21h_write;		{ output text }
          GFXDRVSetCursorPos(bGFXDRVCurrentX, bGFXDRVCurrentY);	{ set the cursor accordingly }
	end;
  end;
  asm
    mov ax, fs
    mov ds, ax
    pop fs
  end;
end;

procedure _int_unhandled;[public, alias:'_INT_UNHANDLED']; interrupt;
begin
  // hlt cpu
  asm
  cli
  hlt
  end;
end;


{******************************************************************************
 *  procedure set_interrupt
 *
 ******************************************************************************
 *
 *  this procedure sets Interrupt's Service Routine to given Address
 *
 ******************************************************************************
 *
 *  INTERRUPT GATE:
 *
 *  8 bytes long (2 LongInt's)
 *
 *  | byte 7 | byte 6 | byte 5 | byte 4 |
 *  | 31-24  | 23-16  |        |        |
 *  +--------+--------+--------+--------+
 *  |  offset  31:16  | access |   00   |
 *  +--------+--------+--------+--------+
 *
 *  | byte 3 | byte 2 | byte 1 | byte 0 |
 *  | 31-24  | 23-16  |        |        |
 *  +--------+--------+--------+--------+
 *  |  selector 15:0  |   offset 15:0   |
 *  +--------+--------+--------+--------+
 *
 *  access (byte 5):
 *
 *  | bit 7  | bit 6  | bit 5  | bit 4  | bit 3:0 |
 *  +--------+--------+--------+--------+---------+
 *  |Present |    Privilege    |   0    |  Type   |
 *  +--------+--------+--------+--------+---------+
 *
 ******************************************************************************}

procedure set_interrupt( Interrupt: byte; Address: Pointer);[public, alias:'SET_INTERRUPT'];
var
   IntOffset: Pointer;
   int:       int_gate;
   tmp:       String;
begin
     IntOffset := Pointer(longint(IDTBASE) + Interrupt * 8);

     int.offset_high := high(DWord(address));
     int.offset_low := low(DWord(address));
     int.access := $8E00;
     int.selector := $8;

     asm
          cli
     end;
          mem_copy( @int, IntOffset, sizeof(int_gate));
     asm
          sti
     end;
end;

procedure init_idt;[public, alias:'INIT_IDT'];
begin
  setup_idt(@_INT_UNHANDLED, @_INT21H);					{ assembler function --> load_idt.s }
end;

begin
end.
