{****************************************************************************************
 * PrepareFpcNasmfiles									*
 *											*
 * strips EXTERNALs from NASM files							*
 *											*
 * written by Bastian Gloeckle,	programmer of nucleOS					*
 *											*
 * Copyright (C) 2003 by nucleOS group							*
 *											*
 * This program is free software; you can redistribute it and/or modify it under the	*
 * terms of the GNU General Public License as published by the Free Software Foundation	*
 * (version 2, June 1991)								*
 *											*
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY	*
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.		*
 *											*
 * You should have received a copy of the GNU General Public License along with this	*
 * program; if not, write to: 								*
 * Free Software Foundation, Inc.							*
 * 59 Temple Place, Suite 330								*
 * Boston, MA 02111-1307 USA								*
 *											*
 *											*
 * You can contanct me by electronic mail: admin@saint-soft.de				*
 ****************************************************************************************}
program pfn;
uses crt, sysutils, pfn_types;

var	Parameters	: TParameters;
	aUnitToDo	: array [1..100] of PUnitToDo;	{ units that must be opened }
	iUnitToDoCnt	: integer;			{ counter for aUnitToDo }

const version='v1.0b2';


{************************************************************************
 * function GetParameters:TParameters;					*
 *									*
 * gets all parameters passed to SE and	saves them in a TParameters	*
 * structure								*
 ************************************************************************}
function GetParameters:TParameters;
var	iCnt	: integer;
	sHelp	: string;
begin
  GetParameters.bHelp		:= false;
  GetParameters.bBits32		:= true;
  GetParameters.bJump		:= true;
  GetParameters.sEntry		:= 'PASCALMAIN';
  GetParameters.sInput		:= '';
  GetParameters.sOutput		:= 'out.s';
  GetParameters.sTempfile	:= 'pfn.tmp';
  GetParameters.sUnitPath	:= '';
  GetParameters.sUnitExtension	:= 's';
  if paramcount=0 then exit;	{ if no parameters have been passed }
  iCnt:=0;
  while (iCnt<paramcount) do
  begin
    inc(iCnt);
    sHelp:=paramstr(iCnt);
    if sHelp[1]<>'-' then	{ is parameter = input file? }
    begin
      GetParameters.sInput:=copy(sHelp,1,length(sHelp));
    end else
    begin
      case sHelp[2] of		{ check character after "-" }
        'o': GetParameters.sOutput	:=copy(sHelp,3,length(sHelp));
	'h': GetParameters.bHelp	:=true;
	'b': GetParameters.bBits32	:=false;
	'e': GetParameters.sEntry	:=copy(sHelp,3,length(sHelp));
	'j': GetParameters.bJump	:=false;
        't': GetParameters.sTempfile	:=copy(sHelp,3,length(sHelp));
        'u': begin
	       GetParameters.sUnitPath	:=copy(sHelp,3,length(sHelp));
	       if GetParameters.sUnitPath[length(GetParameters.sUnitPath)]<>'\' then	{ make sure there is a trailing \ }
	         GetParameters.sUnitPath:=GetParameters.sUnitPath+'\';
	     end;
        'f': GetParameters.sUnitExtension:=copy(sHelp,3,length(sHelp));
      end;
    end;
  end;
end;

{************************************************************************
 * procedure CheckFile(Parameters: TParameters; sFileToCheck: string);	*
 *									*
 * checks a file which units and procedures are needed and updates	*
 * aUnitToDo (recursive procedure!)					*
 ************************************************************************}
procedure CheckFile(Parameters: TParameters; sFileToCheck: string);
var	iCnt		: integer;
	tFilesHndl	: textfile;
	sLine		: string;
	sBuf		: string;
	sFunction	: string;
	bFound		: boolean;
	iUnit		: integer;
begin
  assign(tFileSHndl,sFileToCheck);
  reset(tFileSHndl);
  while (not Eof(tFileSHndl)) do
  begin
    readln(tFileSHndl, sLine);
    if (Upcase(copy(sLine,0,6))='EXTERN') then
    begin
      if (copy(sLine,8,4)='FPC_') or (copy(sLine,8,4)='INIT') then 
        continue;							{ if it is an EXTERN pointing to FPC: skip this}
      sBuf	:= copy(sLine,8,length(sLine));
      sBuf	:= Upcase(copy(trim(sBuf),2,pos('$',sBuf)-2));		{ name of the unit (its the string before the "$$") }
      sFunction := trim(copy(sLine,8,length(sLine)));
      iCnt	:= 0;
      bFound:=false;
      while (iCnt<iUnitToDoCnt) do					{ check if unit is already in aUnitToDo }
      begin
        inc(iCnt);
	if (sBuf=aUnitToDo[iCnt]^.sName) then
	begin
	  bFound:=true;
	  break;
	end;
      end;
      if bFound then							{ if the unit is already in the list }
      begin
        iUnit	:= iCnt;
        iCnt	:= 0;
	bFound	:= false;
	while (iCnt<aUnitToDo[iUnit]^.iFunction) do			{ check if function is already in the list }
	begin
	  inc(iCnt);
	  if (aUnitToDo[iUnit]^.aFunction[iCnt]=sFunction) then
	  begin
	    bFound:=true;
	    break;
	  end;
	end;
	if not bFound then						{ if not -> add it! }
	begin
	  inc(aUnitToDo[iUnit]^.iFunction);
	  aUnitToDo[iUnit]^.aFunction[aUnitToDo[iUnit]^.iFunction] := sFunction;
	end;
      end else
      begin								{ if unit is not in the list, add it, add the function and check the new unit! }
        inc(iUnitToDoCnt);
	New(aUnitToDo[iUnitToDoCnt]);
	aUnitToDo[iUnitToDoCnt]^.sName		:= sBuf;
	aUnitToDo[iUnitToDoCnt]^.iFunction	:= 1;
	aUnitToDo[iUnitToDoCnt]^.aFunction[1]	:= sFunction;
	CheckFile(Parameters,Parameters.sUnitPath+sBuf+'.'+Parameters.sUnitExtension);			{ recursive! }
      end;
    end;
  end;
  close(tFileSHndl);
end;

{************************************************************************
 * procedure AddReturns(Parameters: TParameters);			*
 *									*
 * adds the "RET"s for the EXTERNALs which refer to FPC.		*
 ************************************************************************}
procedure AddReturns(Parameters: TParameters);
var	tFileDHndl	: textfile;
	tFileSHndl	: textfile;
	sLine		: string;
	sUnit		: string;
	sFunction	: string;
	bFound		: boolean;
	iUnitCnt	: integer;
	iFunctionCnt	: integer;
begin
  assign(tFileDHndl,Parameters.sTempfile);
  rewrite(tFileDHndl);
  assign(tFileSHndl,Parameters.sInput);
  reset(tFileSHndl);
  while not Eof(tFileSHndl) do
  begin
    readln(tFileSHndl, sLine);
    if (Upcase(copy(sLine,0,6))='EXTERN') then		{ skip all the EXTERNs (they are in the temp file) }
    begin
      iUnitCnt	:= 0;
      bFound	:= false;
      sUnit	:= copy(sLine,8,length(sLine));
      sUnit	:= Upcase(copy(trim(sUnit),2,pos('$',sUnit)-2));		{ name of the unit (its the string before the "$$") }
      sFunction := trim(copy(sLine,8,length(sLine)));
      while (iUnitCnt<iUnitToDoCnt) do			{ check if this unit is in the list }
      begin
        inc(iUnitCnt);
        if (aUnitToDo[iUnitCnt]^.sName = sUnit) then
        begin
          iFunctionCnt := 0;
	  while (iFunctionCnt<aUnitToDo[iUnitCnt]^.iFunction) do	{ if unit is in list, check if function is in it }
	  begin
	    inc(iFunctionCnt);
	    if (aUnitToDo[iUnitCnt]^.aFunction[iFunctionCnt]=sFunction) then
	    begin
	      bFound := true;
	      break;
	    end;
	  end;
	  break;
        end;
        if bFound then break;
      end;
      if not bFound then 
        writeln(tFileDHndl, sFunction+': ret');				{ if function is not in the list, RET the label }
    end;
  end;
  close(tFileDHndl);
  close(tFileSHndl);
end;

{************************************************************************
 * procedure PrepareUnits(Parameters:TParameters)			*
 *									*
 * generates the (temporary) file, that includes the NASM source lines, *
 * which must be added before "GLOBAL _main". It uses the data in       *
 * aUnitToDo.								*
 ************************************************************************}
procedure PrepareUnits(Parameters:TParameters);
var	tFileTHndl	: textfile;	{ file handle for temporary file }
	tFileSHndl	: textfile;	{ file handle for every source file }
	iUnitCnt	: integer;	{ counter for units }
	iFunctionCnt	: integer;	{ counter for functions }
	sUnitPath	: string;	{ complete path to unit }
	sLine		: string;	{ actual line read }
	sBuf		: string;	{ some buffer }
	iCnt		: integer;	{ come counter }
begin
  assign(tFileTHndl,Parameters.sTempfile);
  append(tFileTHndl);
  iUnitCnt:=0;
  while (iUnitCnt<iUnitToDoCnt) do
  begin
    inc(iUnitCnt);
    if (aUnitToDo[iUnitCnt]^.iFunction=0) then 
      continue;							{ if there are no functions needed from this unit; should never occur }
    writeln('Working on unit: '+aUnitToDo[iUnitCnt]^.sName);
    sUnitPath:=Parameters.sUnitPath+aUnitToDo[iUnitCnt]^.sName+'.'+Parameters.sUnitExtension;
    writeln(tFileTHndl,'; ................................... START code from unit '+aUnitToDo[iUnitCnt]^.sName);
    assign(tFileSHndl,sUnitPath);
    reset(tFileSHndl);
    iCnt:=0;
    while (iCnt<aUnitToDo[iUnitCnt]^.iFunction) and not (Eof(tFileSHndl)) do	{ do until all functions are found or eof is found }
    begin
      readln(tFileSHndl,sLine);
      if Upcase(trim(copy(trim(sLine),0,6)))='GLOBAL' then
      begin
        sBuf		:= trim(copy(trim(sLine),7,length(sLine)));		{ gets the function name }
        iFunctionCnt	:= 0;
	while (iFunctionCnt<aUnitToDo[iUnitCnt]^.iFunction) do
	begin
	  inc(iFunctionCnt);
	  if aUnitToDo[iUnitCnt]^.aFunction[iFunctionCnt] = sBuf then
	  begin
	    sBuf := sLine;							{ be shure to copy the whole line (not only the functionname) }
	    while true do							{ write everthing after the label until an "ALIGN" is found into the tempfile }
	    begin
              writeln(tFileTHndl,sBuf);
	      if Eof(tFileSHndl) then break;
	      if (Upcase(copy(trim(sBuf),0,5))='ALIGN') then break;
	      readln(tFileSHndl,sBuf);
	    end;
	    inc(iCnt);								{ increase iCnt so we know how much functions we've found yet }
	    break;
	  end;
	end;
      end;
    end;
    writeln(tFileTHndl,'; ................................... END code from unit '+aUnitToDo[iUnitCnt]^.sName);
    close(tFileSHndl);
    Dispose(aUnitToDo[iUnitCnt]);
  end;
  close(tFileTHndl);
end;

{************************************************************************
 * procedure Proceed(Parameters:TParameters)				*
 *									*
 * writes the data of the tempfile and the mainfile into the output	*
 * file									*
 ************************************************************************}
procedure Proceed(Parameters:TParameters);
var	tFileSHndl	: textfile;	{ Sourcefile handle }
	tFileDHndl	: textfile;	{ Destinationfile handle }
	tFileTHndl	: textfile;	{ temporary file handle }
	sLine		: string;
	sBuf		: string;
begin
  assign(tFileSHndl,Parameters.sInput);	{ open files }
  reset(tFileSHndl);
  assign(tFileDHndl,Parameters.sOutput);
  rewrite(tFileDHndl);
  if Parameters.bJump then
    writeln(tFileDHndl,'JMP '+Parameters.sEntry);	{ set the jump to the entry point }
  while (not Eof(tFileSHndl)) do
  begin
    readln(tFileSHndl,sLine);
    if (trim(sLine)='BITS 32') and (Parameters.bBits32) then 
      readln(tFileSHndl,sLine);				{ delete "BITS 32" line if user wants to }
    while (Upcase(copy(sLine,0,6))='EXTERN') do		{ skip all the EXTERNs (they are in the temp file) }
      readln(tFileSHndl,sLine); 
    if (trim(sLine)='GLOBAL _main') then			{ print temp file before "GLOBAL _main" }
    begin
      writeln(tFileDHndl,'; :::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::: PFN START');
      assign(tFileTHndl,Parameters.sTempfile);
      reset(tFileTHndl);
      while (not Eof(tFileTHndl)) do
      begin
        readln(tFileTHndl,sBuf);
	writeln(tFileDHndl,sBuf);
      end;
      close(tFileTHndl);
      DeleteFile(Parameters.sTempfile);
      writeln(tFileDHndl,'; :::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::: PFN END');
    end;
    writeln(tFileDHndl,sLine);
  end;
  close(tFileSHndl);
  close(tFileDHndl);
end;

{************************************************************************
 * procedure ShowHelp							*
 *									*
 * displays the help							*
 ************************************************************************}
procedure ShowHelp;
begin
  writeln('Change a FPC generated NASM file so that it can be assembeled to a flat binary');
  writeln('written for FPC v1.0.10 and NASM v0.98.36');
  writeln('');
  writeln('Usage: pfn [options] [file]');
  writeln('');
  writeln('Options:');
  writeln('         -b        do NOT delete the "BITS 32" line');
  writeln('         -e<entry> set the entry point of the file (standard: "PASCALMAIN")');
  writeln('         -f<ext>   set fileextension of units to .<ext> (standard: "s")');
  writeln('         -j        do NOT add the jump to <entry> at the beginning of the file');
  writeln('         -o<file>  output to <file> (standard: "out.s")');
  writeln('         -t<file>  use <file> as temporary file (standard: "pfn.tmp")');
  writeln('         -u<path>  use <path> to look for units (standard: empty)');
  writeln('File:');
  writeln('         the input file');
  writeln('');
end;

{************************************************************************
 * MAIN PROGRAM								*
 ************************************************************************}

begin
  Parameters:=GetParameters;
  writeln('PrepareFpcNasmfile '+version+', written by Bastian Gloeckle (nucleOS group)');
  writeln('This program is published under the GNU General Public License (GPL) version 2');
  writeln('');
  if Parameters.bHelp or (paramcount=0) then				{ Show help? }
  begin
    ShowHelp;
    exit;
  end;
  if not fileExists(Parameters.sInput) then
  begin
    writeln('ERROR: input file does not exist!');
    exit;
  end;
  writeln('Processing: '+Parameters.sInput);
  writeln('Searching for needed units and functions...');
  iUnitToDoCnt:=0;
  CheckFile(Parameters,Parameters.sInput);				{ Search for all needed units and functions. recursive! }
  AddReturns(Parameters);						{ for all EXTERNs, which point on not needed functions, are RETed (into tempfile) }
  PrepareUnits(Parameters);						{ Puts the needed functions all in tempfile }
  Proceed(Parameters);							{ writes mainfile and tempfile together to outputfile }
end.
